/*
 * cDebugLayer.cpp
 *
 * Layer used for debugging purposes.
 */
#include "cDebugLayer.h"
#include "Util/cRegisterList.h"
#include "Util/gError.h"
#include "Endpoint/cEndpoint.h"

#include <stdio.h>
#include <stdlib.h>
#include <time.h>

/*
 * cDebugLayer::Deliver()
 *
 * Purpose:	Delivery interface function.
 * IN:		sender		-> The sender of the msg
 *			buffer		-> The buffer delivered up.
 *			size		-> The size of the buffer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cDebugLayer::Deliver(cEndpoint* sender, cMsgBuffer* buffer, int messageType)
{
	int		r = rand();
	char *output;
	int	  size;
	cIterator*	iter;
	cDeliver*	deliver;

	// Drop this message if necessary.
	if(r < mParam.mProbDropIncomingMsg)
	{
		mDroppedIn++;
		if(mParam.mPrintDrop)
		{
			cerr << "|" << mName << ": INCOMING MSG DROPPED|" << endl;
		}
		return true;
	}
	else
	{
		mPassedIn++;
	}

	// Print the message if necessary.
	if(mParam.mPrintIncomingMsgs)
	{
		// Print the message
		buffer->GetPayload((void **)&output, &size);
		printf("-------%s: INCOMING MESSAGE (Type=%d, size=%d)-------------\n", mName, messageType, size);
		sender->Print(cerr);
		for(int i = 0; i < size; i++)
		{
			printf("%c", output[i]);
		}
		printf("\n---------------------------------------------------\n");
	}

	// Deliver to callbacks.
	iter = mRegisterList->GetIterator();
	while(!iter->Done())
	{
		deliver = (cDeliver *)iter->GetData();
		deliver->Deliver(sender, buffer, messageType);
		iter->GetNext();
	}
	return true;
}

/*
 * cDebugLayer::Init()
 *
 * Purpose:	Initializes the debug layer.
 * IN:		layerBelow	-> The layer below this layer.
 *			param		-> The parameters intended for this layer.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cDebugLayer::Init(cLayer* layerBelow, cParam* param)
{
	cLayerInfo*	belowInfo;
	mRegisterList = NULL;
	cDebugLayerParam*	dParam = (cDebugLayerParam *)param;

	mParam = *dParam;
	strcpy(mName, dParam->mName);

	// Stats
	mDroppedIn = 0;
	mPassedIn = 0;
	mDroppedOut = 0;
	mPassedOut = 0;

	// Seed random number generator.
	srand( (unsigned)time( NULL ) );

	// Set up register list.
	mRegisterList = new cRegisterList(8);	// CHANGE!!!
	if(!mRegisterList)
	{
		return false;
	}

	// Set up layer info.
	mLayerBelow = layerBelow;
	if(mLayerBelow)
	{
		belowInfo = mLayerBelow->GetLayerInfo();
		mLayerInfo = *belowInfo;
	}
	mLayerInfo.SetLayerBelow(layerBelow);

	// Set up connection to layer below
	if(mLayerBelow)
	{
		if(!layerBelow->RegisterDeliverCallback(&mLayerBelowHandle, this))
		{
			gError("Unable to register with layer below.", __LINE__, __FILE__);
			return false;
		}
	}

	return true;
}

/*
 * cDebugLayer::Cleanup()
 *
 * Purpose:	Cleans up the debug layer.
 * IN:		-
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is no longer useable.
 * Return:	true if success, else false.
 */
bool cDebugLayer::Cleanup()
{
	// Remove self from layer below.
	if(mLayerBelow)
	{
		mLayerBelow->UnregisterDeliverCallback(mLayerBelowHandle);
	}

	// Delete the register list.
	if(mRegisterList)
	{
		delete mRegisterList;
		mRegisterList = NULL;
	}
	return true;
}

/*
 * cDebugLayer::Send()
 *
 * Purpose:	Prints out the msg to send and passes it to layer below.
 * IN:		dest		-> The message destination.
 *			buffer		-> the actual message.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The layer is ready to rock.
 * Return:	true if success, else false.
 */
bool cDebugLayer::Send(cGroup* dest, cMsgBuffer* buffer, int messageType)	
{ 
	int r = rand();
	char *output;
	int	  size;

	// Drop this message if necessary.
	if(r < mParam.mProbDropOutgoingMsg)
	{
		mDroppedOut++;
		if(mParam.mPrintDrop)
		{
			cerr << "|" << mName << ": OUTGOING MSG DROPPED|" << endl;
		}
		return true;
	}
	else
	{
		mPassedOut++;
	}

	// Print the message if necessary.
	if(mParam.mPrintIncomingMsgs)
	{
		buffer->GetPayload((void **)&output, &size);
		printf("[%s]: Send>>", mName);
		for(int i = 0; i < size; i++)
		{
			printf("%c", output[i]);
		}
		printf("\n[%s]: Size>>%d\n", mName, size);
	}

	// Send to layer below.
	if(mLayerBelow)
	{
		mLayerBelow->Send(dest, buffer, messageType);
	}

	return true; 
}

/*
 * cDebugLayer::RegisterDeliverCallback()
 *
 * Purpose:	Registers the deliver callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool	cDebugLayer::RegisterDeliverCallback(cHandle* handle, cDeliver* callback) 
{ 
	handle->mLayer = this;
	return mRegisterList->AddObject(handle, (cObject *)callback);
}

/*
 * cDebugLayer::UnregisterDeliverCallback()
 *
 * Purpose:	Unregisters the deliver callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cDebugLayer::UnregisterDeliverCallback(cHandle handle)	
{ 
	if(handle.mLayer == this)
	{
		return mRegisterList->RemoveObject(handle);
	}
	else if(mLayerBelow)
	{
		return mLayerBelow->UnregisterDeliverCallback(handle);
	}
	else
	{
		return false;
	}
}

/*
 * cDebugLayer::RegisterErrorCallback()
 *
 * Purpose:	Registers the asynchronous error callback.
 * IN:		callback	-> The callback function to register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is registered.
 * Return:	The handle used for unregistering.
 */
bool cDebugLayer::RegisterErrorCallback(cHandle* handle, cErrorCallback* callback)
{ 
	if(mLayerBelow)
	{
		return mLayerBelow->RegisterErrorCallback(handle, callback);
	}
	else
	{
		return false;
	}
}

/*
 * cDebugLayer::UnregisterErrorCallback()
 *
 * Purpose:	Unregisters the error callback.
 * IN:		handle	-> The handle that was received at register.
 * OUT:		-
 * Cond:	-
 * PostCnd:	The callback function is removed.
 * Return:	true if success, else false.
 */
bool cDebugLayer::UnregisterErrorCallback(cHandle handle)
{ 
	if(mLayerBelow)
	{
		return mLayerBelow->UnregisterErrorCallback(handle);
	}
	else
	{
		return false;
	}
}

bool	cDebugLayer::RegisterViewCallback(cHandle* handle, cView* callback)
{
	if(mLayerBelow)
	{
		return mLayerBelow->RegisterViewCallback(handle, callback);
	}
	else
	{
		return false;
	}
}
bool	cDebugLayer::UnregisterViewCallback(cHandle handle)
{
	if(mLayerBelow)
	{
		return mLayerBelow->UnregisterViewCallback(handle);
	}
	else
	{
		return false;
	}
}

void cDebugLayer::ReportStats(ostream &stream)
{
	stream << "$$$$$$$$$$$$$$$$$$$$$$$$" << endl;
	stream << mName << " Layer Stats    " << endl;
	stream << "In Drop:   " << mDroppedIn << endl;
	stream << "In Pass:   " << mPassedIn  << endl;
	stream << "Total In:  " << mDroppedIn + mPassedIn << endl;
	stream << "Out Drop:  " << mDroppedOut << endl;
	stream << "Out Pass:  " << mPassedOut  << endl;
	stream << "Total Out: " << mDroppedOut + mPassedOut << endl;
	stream << "$$$$$$$$$$$$$$$$$$$$$$$$" << endl;	
}